-- name: Quality of Life Moveset
-- incompatible: moveset
-- description: Quality of Life Moveset. Features: Ground Pound Jump, Ground Pound Dive, Ground Pound Jump Dive, Twirl Pound, Twirl Pound Jump, Water Pound, Water Pound Jump.\nCreator: TheBoiJoshua.rs\nCo-Author: EmeraldLockdown

local ACT_WATER_GROUND_POUND = ACT_GROUP_SUBMERGED | allocate_mario_action(ACT_FLAG_SWIMMING)
local ACT_WATER_TRIPLE       = ACT_GROUP_SUBMERGED | allocate_mario_action(ACT_FLAG_SWIMMING)
local ACT_WALL_SLIDE         = ACT_GROUP_AIRBORNE  | allocate_mario_action(ACT_FLAG_AIR | ACT_FLAG_MOVING | ACT_FLAG_ALLOW_VERTICAL_WIND_ACTION)

local function mario_update(m)

    if m.action == ACT_GROUND_POUND_LAND and m.controller.buttonPressed & A_BUTTON ~= 0 then
        m.vel.y = 30
        set_mario_action(m, ACT_TRIPLE_JUMP, 0)
    end

    if m.action == ACT_TWIRLING and m.controller.buttonDown & Z_TRIG ~= 0 then
        m.vel.y = -50
        spawn_mist_particles_variable(0.1, 0, 11)
    end

    if m.action == ACT_TWIRL_LAND and m.controller.buttonDown & Z_TRIG ~= 0 and m.actionTimer == 0 then
        set_mario_action(m, ACT_GROUND_POUND, 0)
        m.actionTimer = 5000
    end

    if m.action == ACT_TWIRL_LAND then
        m.actionTimer = m.actionTimer + 1
    end

    if m.action == ACT_GROUND_POUND and m.controller.buttonPressed & B_BUTTON ~= 0 then
        m.vel.y = 36
        m.forwardVel = 25
        m.faceAngle.y = m.intendedYaw
        m.particleFlags = m.particleFlags | PARTICLE_MIST_CIRCLE
        set_mario_action(m, ACT_DIVE, 0)
    end

    if m.action == ACT_WATER_JUMP and m.controller.buttonPressed & Z_TRIG ~= 0 then
        set_mario_action(m, ACT_GROUND_POUND, 0)
    end

    if m.action == ACT_STEEP_JUMP and m.controller.buttonPressed & Z_TRIG ~= 0 then
        set_mario_action(m, ACT_GROUND_POUND, 0)
    end

    if (m.action == ACT_FORWARD_ROLLOUT or m.action == ACT_BACKWARD_ROLLOUT
    or m.action == ACT_FALLING_EXIT_AIRBORNE) and m.controller.buttonPressed & Z_TRIG ~= 0 then
        set_mario_action(m, ACT_GROUND_POUND, 0)
    end

    if m.action & ACT_FLAG_SWIMMING ~= 0 and m.controller.buttonPressed & Z_TRIG ~= 0
    and m.action ~= ACT_WATER_GROUND_POUND then
        set_mario_action(m, ACT_WATER_GROUND_POUND, 0)
    end

end

---@param m MarioState
local function before_mario_update(m)
    if m.prevAction == ACT_GROUND_POUND_LAND and m.action == ACT_TRIPLE_JUMP and m.controller.buttonPressed & B_BUTTON ~= 0 then
        m.vel.y = 36
        m.forwardVel = 25
        m.faceAngle.y = m.intendedYaw
        m.particleFlags = m.particleFlags | PARTICLE_MIST_CIRCLE
        set_mario_action(m, ACT_DIVE, 0)
    end
end

---@param m MarioState
local function act_water_ground_pound(m)

    -- lots of code taken from the ground pound action
    m.actionTimer = m.actionTimer + 1

    local stepResult = perform_water_step(m)

    m.vel.x = 0
    m.vel.z = 0
    m.forwardVel = 0
    m.faceAngle.x = 0
    m.faceAngle.z = 0

    if m.actionState == 0 then

        m.vel.y = 0

        if m.actionTimer < 10 then
            local yOffset = 20 - 2 * m.actionTimer
            if m.pos.y + yOffset + 160.0 < m.ceilHeight then
                m.pos.y = m.pos.y + yOffset
                m.peakHeight = m.pos.y
                vec3f_copy(m.marioObj.header.gfx.pos, m.pos)
            end
        end

        set_mario_animation(m, MARIO_ANIM_START_GROUND_POUND)
        if (m.actionTimer == 0) then
            play_sound(SOUND_ACTION_SPIN, m.marioObj.header.gfx.cameraToObject)
        end

        m.actionTimer = m.actionTimer + 1
        if (m.actionTimer >= m.marioObj.header.gfx.animInfo.curAnim.loopEnd + 4) then
            m.actionState = 1
            m.actionTimer = 0
        end
    elseif m.actionState == 1 then
        if m.actionTimer < 0.5 * 30 then
            m.vel.y = m.vel.y - 5
        elseif m.action >= 0.5 * 30 then
            m.vel.y = m.vel.y + 5
        end

        m.vel.y = clampf(m.vel.y, -50, 0)

        set_mario_animation(m, MARIO_ANIM_GROUND_POUND)

        if m.controller.buttonPressed & B_BUTTON ~= 0 then
            m.forwardVel = 90
            m.faceAngle.y = m.intendedYaw
            m.input = m.input & ~INPUT_B_PRESSED
            return set_mario_action(m, ACT_BREASTSTROKE, 0)
        end

        if m.actionTimer >= 0.5 * 30 and m.vel.y == 0 then
            m.actionState = 2
        end

        if stepResult == WATER_STEP_HIT_FLOOR then
            play_mario_heavy_landing_sound(m, SOUND_ACTION_TERRAIN_HEAVY_LANDING)
            set_mario_particle_flags(m, (PARTICLE_MIST_CIRCLE | PARTICLE_HORIZONTAL_STAR), 0)
            m.actionState = 2
            m.actionTimer = 0
        end
    else
        m.vel.y = 0

        -- check this to ensure ground pound land isn't instant
        if m.actionTimer >= 0.25 * 30 then
            return set_mario_action(m, ACT_WATER_IDLE, 0)
        else
            -- allow mario to "jump" up
            if m.controller.buttonDown & A_BUTTON ~= 0 then
                set_mario_action(m, ACT_WATER_TRIPLE, 0)
                m.vel.y = 50
            end
        end
    end

    return 0
end

---@param m MarioState
local function act_water_triple(m)
    m.vel.x = 0
    m.vel.z = 0

    m.vel.y = m.vel.y - 1.25
    local curVelY = m.vel.y

    perform_water_step(m)
    set_mario_animation(m, MARIO_ANIM_TRIPLE_JUMP)

    if m.actionTimer > 1.25 * 30 then
        set_mario_action(m, ACT_WATER_IDLE, 0)
    end

    m.actionTimer = m.actionTimer + 1

    -- get out of the water action if we exit water
    if m.pos.y > m.waterLevel - 90 then
        set_mario_action(m, ACT_TRIPLE_JUMP, 1)
        m.pos.y = m.waterLevel
        m.vel.y = curVelY
    end
end

local function act_wall_slide(m)

    if m.controller.buttonPressed & A_BUTTON ~= 0 then
        local returnValue = set_mario_action(m, ACT_WALL_KICK_AIR, 0)

        m.vel.y = 60.0

        if m.forwardVel < 20.0 then
            m.forwardVel = 20.0
        end
        m.wallKickTimer = 0
        return returnValue
    end

    -- attempt to stick to the wall a bit. if it's 0, sometimes you'll get kicked off of slightly sloped walls
    mario_set_forward_vel(m, -1.0)

    m.particleFlags = m.particleFlags | PARTICLE_DUST -- dust particles

    -- play sliding sound
    play_sound(SOUND_MOVING_TERRAIN_SLIDE + m.terrainSoundAddend, m.marioObj.header.gfx.cameraToObject)
    -- set mario's animation to the start of the wallkick
    -- looks like a sliding animation so it works
    set_mario_animation(m, MARIO_ANIM_START_WALLKICK)

    -- if we landed ...
    if perform_air_step(m, 0) == AIR_STEP_LANDED then
        -- ... set mario's vel to 0
        mario_set_forward_vel(m, 0.0)
        -- check fall damage or getting stuck
        if check_fall_damage_or_get_stuck(m, ACT_HARD_BACKWARD_GROUND_KB) == 0 then
            return set_mario_action(m, ACT_FREEFALL_LAND, 0)
        end
    end

    -- increase action timer
    m.actionTimer = m.actionTimer + 1
    -- if the wall doesn't exist and our action timer is more than 2 then set our action to freefall
    if m.wall == nil and m.actionTimer > 2 then
        mario_set_forward_vel(m, 0.0)
        return set_mario_action(m, ACT_FREEFALL, 0)
    end

    -- gravity
    m.vel.y = -25

    return 0
end

---@param m MarioState
local function mario_on_set_action(m)
    -- wall slide
    if m.action == ACT_SOFT_BONK then
        m.faceAngle.y = m.faceAngle.y + 0x8000 -- 90 degrees
        set_mario_action(m, ACT_WALL_SLIDE, 0)
    end

    -- water ground pound
    if m.prevAction == ACT_GROUND_POUND and m.action & ACT_FLAG_SWIMMING ~= 0 then
        set_mario_action(m, ACT_WATER_GROUND_POUND, 0)
        m.actionState = 1
        m.actionTimer = 0.4 * 30
    end
end

-- hooks
hook_event(HOOK_MARIO_UPDATE, mario_update)
hook_event(HOOK_BEFORE_MARIO_UPDATE, before_mario_update)
hook_event(HOOK_ON_SET_MARIO_ACTION, mario_on_set_action)

hook_mario_action(ACT_WALL_SLIDE, act_wall_slide)
hook_mario_action(ACT_WATER_GROUND_POUND, act_water_ground_pound)
hook_mario_action(ACT_WATER_TRIPLE, act_water_triple)